/******************************************************************************
	CSourceDocument.c
	
	Document methods for the source (ada) files


	Copyright (C) 1985-1992  New York University
	Copyright (C) 1994 George Washington University
	 
	This file is part of the GWAdaEd system, an extension of the Ada/Ed-C
	system.  See the Ada/Ed README file for warranty (none) and distribution
	info and also the GNU General Public License for more details.


 ******************************************************************************/

#include <assert.h>
#include <Global.h>
#include <Commands.h>
#include <CApplication.h>
#include <CBartender.h>
#include <TBUtilities.h>
#include <CDesktop.h>
#include <PPCToolBox.h>

#include "CSourceDocument.h"
#include "CListingDocument.h"
#include "CLibrary.h"
#include "CMenuWindow.h"
#include "CTextPane.h"
#include "MacAdaMrgp.h"
#include "CGotoLine.h"

#include "AdaGlobals.h"
#include "CFileMgr.h"
#include "CAdaApp.h"
#include "AppCommands.h"
#include "SFGetFolder.h"
#include "ErrMessages.h"
#include "HelpBalloons.h"

extern	CBartender	*gBartender;	/* The menu handling object */
extern	CDesktop	*gDesktop;		/* The enclosure for all windows */
extern OSType		gSignature;			/* Creator for Application's files	*/

/***
 * ISourceDocument
 *
 ***/
 
void CSourceDocument::ISourceDocument(CApplication *aSupervisor,
	Boolean compile)
{
	CTextDocument::ITextDocument(aSupervisor);
	fReadOnly = false;			// we can edit this one
	itsListing = NULL;
	fCompilingOnly = compile;
}

/****
 *	Close
 *
 *	Try to close itsListing, then call inherited and close this
 *	document.  If itsListing is not closed, then do not close
 *	me.  If its listing is closed, set collaborator to NULL.
 *
 ****/
Boolean		CSourceDocument::Close(Boolean quitting)
{
	if (itsListing) {
		if (itsListing->Close(quitting)) {
			itsListing = NULL;
			return inherited::Close(quitting);
		}
		else
			return false;
	}
	else
		return inherited::Close(quitting);
}

/******************************************************************************
 DoRevert {OVERRIDE}

	Revert to previously saved version of document
 ******************************************************************************/

void CSourceDocument::DoRevert(void)

{
		/* THIS WAS COPIED FROM THE CSAVER CLASS */
		/* We use the all-or-nothing approach to revert, throwing	*/
		/* away the current window before attempting to reread the	*/
		/* saved version.  If this fails, user is left with no		*/
		/* window at all.  On the other hand, it is much more		*/
		/* likely to succeed, as more memory is available.			*/

	ForgetObject(itsWindow);
	itsGopher = this;					/* TCL bug - not reset when window	*/
										/*	 disposed						*/

	if (itsFile != NULL)
		//ReadDocument();					/* Reread the file					*/
		;
	else
		NewFile();						/* Back to square one				*/
}

/****
 *	MakeWindow
 *
 *	Create an instance of the window that stores its name on the
 *	Windows menu.
 *
 ****/

void	CSourceDocument::MakeWindow(void)
{
	itsWindow = new(CMenuWindow);
	itsWindow->IWindow(WINDculture, FALSE, gDesktop, this);

	itsWindow->SetHelpResID(hlpSource);
}

/****
 *	UpdateMenus
 *
 *	Set menu state based on current context of this document
 *
 ****/
		
		static void AbleCmd(CAdaApp *ap, long cmd)
		{
			if (ap->CanCmd(cmd))
				gBartender->EnableCmd(cmd);
			else
				gBartender->DisableCmd(cmd);
		}
		

void	CSourceDocument::UpdateMenus(void)
{

	inherited::UpdateMenus();

	// if we have a listing file, then...
	if (itsListing) {
		if (itsListing->CanCmdNext())
			gBartender->EnableCmd(cmdNextError);
		else
			gBartender->DisableCmd(cmdNextError);
	
		if (itsListing->CanCmdPrev())
			gBartender->EnableCmd(cmdPreviousError);
		else
			gBartender->DisableCmd(cmdPreviousError);
	}

	// Can we compile this file?  Only if the file has at least
	// some text in it.
	if (GetNumLines() > 0) {
		gBartender->EnableCmd(cmdCompile);
		gBartender->EnableCmd(cmdOpenGotoLine);
	}
	else {
		gBartender->DisableCmd(cmdCompile);
		gBartender->DisableCmd(cmdOpenGotoLine);
	}


	gBartender->EnableCmd(cmdPageSetup);
	gBartender->EnableCmd(cmdPrint);
}

/***
 * DoCommand
 *
 ***/
 
void	CSourceDocument::DoCommand(long theCommand)
{

	switch (theCommand) {
		case cmdOpenGotoLine: {
			CGotoLine	*dialog;
			long		dismissCmd;
			long		line;

			line = fTextEdit->GetCursorLine();
			dialog = new CGotoLine;
			dialog->ICGotoLine(this, fTextEdit, line);
			dialog->BeginDialog();

			dismissCmd = dialog->DoModalDialog(cmdOK);

			if (dismissCmd == cmdOK) {
				
				line = dialog->GetLineNumber();
				SelectLine(line);
				ScrollToSelection();
			}
			ForgetObject(dialog);
			break;
		}

		case cmdCompile:
			// Call the application compile method with this as the
			// parameter.
			if (itsListing) {
				if (itsListing->Close(false)) {
					itsListing = NULL;
					gAdaApp->Compile(this);
				}
				else {
					SysBeep(1);
					// put up a message saying, the listing file must
					// be closed before we can compile again...
				}
			}
			else
				gAdaApp->Compile(this);
			break;

		case cmdNextError:		// pass these two to its listing
		case cmdPreviousError:
			assert(itsListing);
			itsListing->DoCommand(theCommand);
			break;

		default:
			inherited::DoCommand(theCommand);
			break;
	}
}

/***
 *	CreateListing
 *
 ***/

void	CSourceDocument::CreateListing(Str255 name)
{
	Str255	fullSrc, fullMsg, fullLis;
	Handle	teHandle;
	Msg *ptr;

	// get files from the CFileMgr and change to c strings
	gFileMgr->GetMsgFile(fullMsg);
	PtoCstr(fullMsg);

	gFileMgr->GetLisFile(fullLis);
	PtoCstr(fullLis);

	// process msg file to produce a lis file
	teHandle = GetTextHandle();
	MoveHHi(teHandle);
	HLock(teHandle);

	ptr = NULL;
	fNumErrors = merge_error_source(teHandle, (char *)fullMsg,
				(char *)fullLis, &ptr);

	fErrorMsgs = ptr;
	HUnlock(teHandle);

	if (fNumErrors < 0) {

		// We couldn't open the listing file due to some internal error.
		Failure(errListingNotCreated, 0);

	}
}

OSErr	CSourceDocument::SendODOC(AEAddressDesc *targetAddress,
				AliasHandle targetFile)
{
	OSErr err;
	AppleEvent theAE,reply;
	AEDescList descList;
	
	reply.dataHandle = nil;
	
	err = AECreateAppleEvent(kCoreEventClass,kAEOpenDocuments,targetAddress,
				kAutoGenerateReturnID,kAnyTransactionID,&theAE);
	if (err!=noErr)
		return err;
	
	err = AECreateList(nil,0,false,&descList);
	if (err!=noErr)
		return err;
	
	HLock((Handle)targetFile);
	err = AEPutPtr(&descList,0,typeAlias,(Ptr)*targetFile,sizeof(AliasHandle)+(**targetFile).aliasSize);
	HUnlock((Handle)targetFile);
	if (err!=noErr)
		return err;
	
	err = AEPutParamDesc(&theAE,keyDirectObject,&descList);
	if (err!=noErr)
		return err;
	
	err = AESend(&theAE,&reply,kAENoReply,kAENormalPriority,kAEDefaultTimeout,nil,nil);
	
	err = AEDisposeDesc(&descList);
	err = AEDisposeDesc(&theAE);
	if (reply.dataHandle)
		err = AEDisposeDesc(&reply);
}

void	CSourceDocument::AEOpenListing(FSSpec fileSpec)
{
	PortInfoRec portInfo;
	AEAddressDesc targetAddress;
	TargetID targetID;
	OSErr err;
	AliasHandle aliasTarget;
	ProcessSerialNumber PSN;

	PSN.highLongOfPSN = 0;
	PSN.lowLongOfPSN = kCurrentProcess;		// use current process
	err = AECreateDesc(typeProcessSerialNumber, (Ptr)&PSN,
		sizeof(ProcessSerialNumber), &targetAddress);

	if (err == noErr) {

		err = NewAlias(nil, &fileSpec, &aliasTarget);

		if (err == noErr) {

			err = SendODOC(&targetAddress, aliasTarget);
			err = AEDisposeDesc(&targetAddress);
			DisposHandle((Handle)aliasTarget);
		}
	}
}

/****
 *	OpenListing
 *
 ****/
void	CSourceDocument::OpenListing(void)
{
	FSSpec	currDoc;
	short 	wdRefNum;
	FInfo	fileInfo;
	Str255	fullLis;
	SFReply	errorsFile;


	// This document is not compile only anymore
	fCompilingOnly = false;

	// To open listing file, we must get the listing name,
	gFileMgr->GetLisFile(fullLis);

	// AEvent way of doing things...
	gSource = this;
	FailOSErr( FSMakeFSSpec(0, 0, fullLis, &currDoc));	
	AEOpenListing(currDoc);

#ifdef OLD_WAY_NEVER_WORKED_COMPLETELY

	assert(gLibrary);		// this has to be true at all times

	// Build a StdFileReply record
	FailOSErr( FSMakeFSSpec(0, 0, fullLis, &currDoc));	
	FailOSErr( OpenWD( currDoc.vRefNum, currDoc.parID,
					gSignature, &wdRefNum));
						
	FailOSErr( FSpGetFInfo( &currDoc, &fileInfo));
		
	errorsFile.good = TRUE;				// this is important to set here
	errorsFile.vRefNum = wdRefNum;
	errorsFile.fType = fileInfo.fdType;
	errorsFile.version = errorsFile.copy = 0;
	CopyPString( currDoc.name, errorsFile.fName);
		

	CView::ForceNextPrepare();

	// (1) this one worked alone: 
	//	Suspend();

	// (2) and this one too
	//theWindow = GetWindow();
	//if (theWindow)
	//	theWindow->Deactivate();

	// But this one is completely generic
	if (FrontWindow()) {
		CWindow *theWindow;

		theWindow = (CWindow *) GetWRefCon(FrontWindow());
		if (member(theWindow, CWindow))
			theWindow->Deactivate();
	}

	// and create the listing document
	itsListing = NULL;
	TRY
	{
		FailNIL( itsListing = new CListingDocument );
		itsListing->IListingDocument(this, fNumErrors, fErrorMsgs);
		itsListing->OpenFile(&errorsFile);
	}
	CATCH
	{
		ForgetObject(itsListing);
		itsListing = NULL;
	}
	ENDTRY;
#endif
}

void	CSourceDocument::ConnectListing(CListingDocument *list)
{
	itsListing = list;
}

/****
 *	ListingClosed
 *
 *	Listing document should call this method when it is closed, so
 *	we can marked this member field appropriately.
 *
 ****/
void	CSourceDocument::ListingClosed(void)
{
	itsListing = NULL;
}

long	CSourceDocument::GetNumErrors(void)
{
	return fNumErrors;
}

Msg		*CSourceDocument::GetErrorMsgs(void)
{
	return fErrorMsgs;
}

/****
 *	IsItCompileOnly
 *
 *	This returns the flag passed in during creation of this document.
 *	The document will be closed after compilation if it was compiled
 *	successfully.
 *
 ****/

Boolean	CSourceDocument::IsItCompileOnly(void)
{
	return fCompilingOnly;
}

